//                              Package : omnithread
// omnithread/nt.cc             Created : 6/95 tjr
//
//    Copyright (C) 2005-2025 Apasphere Ltd
//    Copyright (C) 1995-1999 AT&T Laboratories Cambridge
//
//    This file is part of the omnithread library
//
//    The omnithread library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Lesser General Public
//    License as published by the Free Software Foundation; either
//    version 2.1 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public
//    License along with this library. If not, see http://www.gnu.org/licenses/
//

//
// Implementation of OMNI thread abstraction for NT threads
//

#include <stdlib.h>
#include <errno.h>
#include <omnithread.h>
#include <process.h>

#define DB(x) // x 
//#include <iostream.h> or #include <iostream> if DB is on.

static void get_time_now(omni_time_t::secs_t& abs_sec,
			 omni_time_t::ns_t&   abs_nsec);

///////////////////////////////////////////////////////////////////////////
//
// Mutex
//
///////////////////////////////////////////////////////////////////////////


omni_mutex::omni_mutex(void)
{
  InitializeCriticalSection(&crit);
}

omni_mutex::~omni_mutex(void)
{
  DeleteCriticalSection(&crit);
}



///////////////////////////////////////////////////////////////////////////
//
// Condition variable
//
///////////////////////////////////////////////////////////////////////////

omni_condition::omni_condition(omni_mutex* m) : mutex(m)
{
  InitializeConditionVariable(&cond);
}


omni_condition::~omni_condition()
{
}


void
omni_condition::wait()
{
  SleepConditionVariableCS(&cond, &mutex->crit, INFINITE);
}


int
omni_condition::timedwait(omni_time_t::secs_t abs_sec,
                          omni_time_t::ns_t   abs_nsec)
{
  omni_time_t::secs_t now_sec;
  omni_time_t::ns_t   now_nsec;

  get_time_now(now_sec, now_nsec);

  DWORD timeout;
  if ((abs_sec <= now_sec) && ((abs_sec < now_sec) || (abs_nsec < now_nsec))) {
    timeout = 0;
  }
  else {
    timeout = (abs_sec-now_sec) * 1000;

    if (abs_nsec < now_nsec)
      timeout -= (now_nsec-abs_nsec) / 1000000;
    else
      timeout += (abs_nsec-now_nsec) / 1000000;
  }

  BOOL result = SleepConditionVariableCS(&cond, &mutex->crit, timeout);

  return result ? 1 : 0;
}


void
omni_condition::signal(void)
{
  WakeConditionVariable(&cond);
}


void
omni_condition::broadcast(void)
{
  WakeAllConditionVariable(&cond);
}


///////////////////////////////////////////////////////////////////////////
//
// Counting semaphore
//
///////////////////////////////////////////////////////////////////////////


#define SEMAPHORE_MAX 0x7fffffff


omni_semaphore::omni_semaphore(unsigned int initial)
{
  nt_sem = CreateSemaphore(0, initial, SEMAPHORE_MAX, 0);

  if (nt_sem == 0) {
    DB( cerr << "omni_semaphore::omni_semaphore: CreateSemaphore error "
        << GetLastError() << endl );
    throw omni_thread_fatal(GetLastError());
  }
}


omni_semaphore::~omni_semaphore(void)
{
  if (!CloseHandle(nt_sem)) {
    DB( cerr << "omni_semaphore::~omni_semaphore: CloseHandle error "
        << GetLastError() << endl );
  }
}


void
omni_semaphore::wait(void)
{
  if (WaitForSingleObject(nt_sem, INFINITE) != WAIT_OBJECT_0)
    throw omni_thread_fatal(GetLastError());
}


int
omni_semaphore::trywait(void)
{
  switch (WaitForSingleObject(nt_sem, 0)) {

  case WAIT_OBJECT_0:
    return 1;
  case WAIT_TIMEOUT:
    return 0;
  }

  throw omni_thread_fatal(GetLastError());
  return 0; /* keep msvc++ happy */
}


void
omni_semaphore::post(void)
{
  if (!ReleaseSemaphore(nt_sem, 1, 0))
    throw omni_thread_fatal(GetLastError());
}



///////////////////////////////////////////////////////////////////////////
//
// Thread
//
///////////////////////////////////////////////////////////////////////////


//
// Static variables
//

omni_mutex* omni_thread::next_id_mutex;
int omni_thread::next_id = 0;
static DWORD self_tls_index;

static unsigned int stack_size = 0;

//
// Initialisation function (gets called before any user code).
//

static int& count() {
  static int the_count = 0;
  return the_count;
}

omni_thread::init_t::init_t(void)
{
  if (count()++ != 0) // only do it once however many objects get created.
    return;

  DB(cerr << "omni_thread::init: NT implementation initialising\n");

  self_tls_index = TlsAlloc();

  if (self_tls_index == 0xffffffff)
    throw omni_thread_fatal(GetLastError());

  next_id_mutex = new omni_mutex;

  //
  // Create object for this (i.e. initial) thread.
  //

  omni_thread* t = new omni_thread;

  t->_state = STATE_RUNNING;

  if (!DuplicateHandle(GetCurrentProcess(), GetCurrentThread(),
		       GetCurrentProcess(), &t->handle,
		       0, FALSE, DUPLICATE_SAME_ACCESS))
    throw omni_thread_fatal(GetLastError());

  t->nt_id = GetCurrentThreadId();

  DB(cerr << "initial thread " << t->id() << " NT thread id " << t->nt_id
     << endl);

  if (!TlsSetValue(self_tls_index, (LPVOID)t))
    throw omni_thread_fatal(GetLastError());

  if (!SetThreadPriority(t->handle, nt_priority(PRIORITY_NORMAL)))
    throw omni_thread_fatal(GetLastError());
}

omni_thread::init_t::~init_t(void)
{
  if (--count() != 0) return;

  omni_thread* self = omni_thread::self();
  if (!self) return;

  TlsSetValue(self_tls_index, (LPVOID)0);
  delete self;

  delete next_id_mutex;

  TlsFree(self_tls_index);
}

//
// Wrapper for thread creation.
//

extern "C" 
#ifndef __BCPLUSPLUS__
unsigned __stdcall
#else
void _USERENTRY
#endif
omni_thread_wrapper(void* ptr)
{
  omni_thread* me = (omni_thread*)ptr;

  DB(cerr << "omni_thread_wrapper: thread " << me->id()
     << " started\n");

  if (!TlsSetValue(self_tls_index, (LPVOID)me))
    throw omni_thread_fatal(GetLastError());

  //
  // Now invoke the thread function with the given argument.
  //

  if (me->fn_void != 0) {
    (*me->fn_void)(me->thread_arg);
    omni_thread::exit();
  }

  if (me->fn_ret != 0) {
    void* return_value = (*me->fn_ret)(me->thread_arg);
    omni_thread::exit(return_value);
  }

  if (me->detached) {
    me->run(me->thread_arg);
    omni_thread::exit();
  } else {
    void* return_value = me->run_undetached(me->thread_arg);
    omni_thread::exit(return_value);
  }

  // should never get here.
#ifndef __BCPLUSPLUS__
  return 0;
#endif
}


//
// Constructors for omni_thread - set up the thread object but don't
// start it running.
//

// construct a detached thread running a given function.

omni_thread::omni_thread(void (*fn)(void*), void* arg, priority_t pri)
{
  common_constructor(arg, pri, 1);
  fn_void = fn;
  fn_ret = 0;
}

// construct an undetached thread running a given function.

omni_thread::omni_thread(void* (*fn)(void*), void* arg, priority_t pri)
{
  common_constructor(arg, pri, 0);
  fn_void = 0;
  fn_ret = fn;
}

// construct a thread which will run either run() or run_undetached().

omni_thread::omni_thread(void* arg, priority_t pri)
{
    common_constructor(arg, pri, 1);
    fn_void = 0;
    fn_ret = 0;
}

// common part of all constructors.

void
omni_thread::common_constructor(void* arg, priority_t pri, int det)
{
  _state = STATE_NEW;
  _priority = pri;

  next_id_mutex->lock();
  _id = next_id++;
  next_id_mutex->unlock();

  thread_arg = arg;
  detached = det;     // may be altered in start_undetached()

  cond_semaphore = CreateSemaphore(0, 0, SEMAPHORE_MAX, 0);

  if (cond_semaphore == 0)
    throw omni_thread_fatal(GetLastError());

  cond_next = cond_prev = 0;
  cond_waiting = FALSE;

  handle = 0;

  _dummy       = 0;
  _values      = 0;
  _value_alloc = 0;
}


//
// Destructor for omni_thread.
//

omni_thread::~omni_thread(void)
{
  DB(cerr << "destructor called for thread " << id() << endl);
  if (_values) {
    for (key_t i=0; i < _value_alloc; i++) {
      if (_values[i]) {
	delete _values[i];
      }
    }
    delete [] _values;
  }
  if (handle)
    CloseHandle(handle);

  if (cond_semaphore)
    CloseHandle(cond_semaphore);
}


//
// Start the thread
//

void
omni_thread::start(void)
{
  omni_mutex_lock l(mutex);

  if (_state != STATE_NEW)
    throw omni_thread_invalid();

#ifndef __BCPLUSPLUS__
  // MSVC++ or compatiable
  unsigned int t;
  handle = (HANDLE)_beginthreadex(0,
				  stack_size,
				  omni_thread_wrapper,
				  (LPVOID)this,
				  CREATE_SUSPENDED, 
				  &t);
  nt_id = t;
  if (handle == 0)
    throw omni_thread_fatal(GetLastError());
#else
  // Borland C++
  handle = (HANDLE)_beginthreadNT(omni_thread_wrapper,
				  stack_size,
				  (void*)this,
				  0,
				  CREATE_SUSPENDED,
				  &nt_id);
  if (handle == INVALID_HANDLE_VALUE)
    throw omni_thread_fatal(errno);
#endif

  if (!SetThreadPriority(handle, nt_priority(_priority)))
    throw omni_thread_fatal(GetLastError());

  if (ResumeThread(handle) == 0xffffffff)
    throw omni_thread_fatal(GetLastError());

  _state = STATE_RUNNING;
}


//
// Start a thread which will run the member function run_undetached().
//

void
omni_thread::start_undetached(void)
{
  if ((fn_void != 0) || (fn_ret != 0))
    throw omni_thread_invalid();

  detached = 0;
  start();
}


//
// join - simply check error conditions & call WaitForSingleObject.
//

void
omni_thread::join(void** status)
{
  mutex.lock();

  if ((_state != STATE_RUNNING) && (_state != STATE_TERMINATED)) {
    mutex.unlock();
    throw omni_thread_invalid();
  }

  mutex.unlock();

  if (this == self())
    throw omni_thread_invalid();

  if (detached)
    throw omni_thread_invalid();

  DB(cerr << "omni_thread::join: doing WaitForSingleObject\n");

  if (WaitForSingleObject(handle, INFINITE) != WAIT_OBJECT_0)
    throw omni_thread_fatal(GetLastError());

  DB(cerr << "omni_thread::join: WaitForSingleObject succeeded\n");

  if (status)
    *status = return_val;

  delete this;
}


//
// Change this thread's priority.
//

void
omni_thread::set_priority(priority_t pri)
{
  omni_mutex_lock l(mutex);

  if (_state != STATE_RUNNING)
    throw omni_thread_invalid();

  _priority = pri;

  if (!SetThreadPriority(handle, nt_priority(pri)))
    throw omni_thread_fatal(GetLastError());
}


//
// create - construct a new thread object and start it running.  Returns thread
// object if successful, null pointer if not.
//

// detached version

omni_thread*
omni_thread::create(void (*fn)(void*), void* arg, priority_t pri)
{
  omni_thread* t = new omni_thread(fn, arg, pri);
  t->start();
  return t;
}

// undetached version

omni_thread*
omni_thread::create(void* (*fn)(void*), void* arg, priority_t pri)
{
  omni_thread* t = new omni_thread(fn, arg, pri);
  t->start();
  return t;
}


//
// exit() _must_ lock the mutex even in the case of a detached thread.  This is
// because a thread may run to completion before the thread that created it has
// had a chance to get out of start().  By locking the mutex we ensure that the
// creating thread must have reached the end of start() before we delete the
// thread object.  Of course, once the call to start() returns, the user can
// still incorrectly refer to the thread object, but that's their problem.
//

void
omni_thread::exit(void* return_value)
{
  omni_thread* me = self();

  if (me) {
    me->mutex.lock();

    me->_state = STATE_TERMINATED;

    me->mutex.unlock();

    DB(cerr << "omni_thread::exit: thread " << me->id() << " detached "
       << me->detached << " return value " << return_value << endl);

    if (me->detached) {
      delete me;
    }
    else {
      me->return_val = return_value;

      if (me->_values) {
	// Delete per-thread state here, to ensure value destructors
	// are executed by this thread.
          
	for (key_t i=0; i < me->_value_alloc; i++) {
	  if (me->_values[i]) {
	    delete me->_values[i];
	  }
	}
	delete [] me->_values;
	me->_values = 0;
	me->_value_alloc = 0;
      }
    }
  }
  else
    {
      DB(cerr <<
	 "omni_thread::exit: called with a non-omnithread. Exit quietly." <<
	 endl);
    }
#ifndef __BCPLUSPLUS__
  // MSVC++ or compatiable
  //   _endthreadex() does not automatically closes the thread handle.
  //   The omni_thread dtor closes the thread handle.
  _endthreadex(0);
#else
  // Borland C++
  //   _endthread() does not automatically closes the thread handle.
  //   _endthreadex() is only available if __MFC_COMPAT__ is defined and
  //   all it does is to call _endthread().
  _endthread();
#endif
}


omni_thread*
omni_thread::self(void)
{
    LPVOID me;

    me = TlsGetValue(self_tls_index);

    if (me == 0) {
      DB(cerr << "omni_thread::self: called with a non-omnithread. NULL is returned." << endl);
    }
    return (omni_thread*)me;
}


unsigned long
omni_thread::plat_id()
{
  return GetCurrentThreadId();
}


void
omni_thread::yield(void)
{
  Sleep(0);
}


#define MAX_SLEEP_SECONDS (DWORD)4294966        // (2**32-2)/1000

void
omni_thread::sleep(omni_time_t::secs_t secs, omni_time_t::ns_t nanosecs)
{
  if (secs <= MAX_SLEEP_SECONDS) {
    Sleep(secs * 1000 + nanosecs / 1000000);
    return;
  }

  DWORD no_of_max_sleeps = secs / MAX_SLEEP_SECONDS;

  for (DWORD i = 0; i < no_of_max_sleeps; i++)
    Sleep(MAX_SLEEP_SECONDS * 1000);

  Sleep((secs % MAX_SLEEP_SECONDS) * 1000 + nanosecs / 1000000);
}


void
omni_thread::get_time(omni_time_t::secs_t* abs_sec,
                      omni_time_t::ns_t*   abs_nsec,
                      omni_time_t::secs_t  rel_sec,
                      omni_time_t::ns_t    rel_nsec)
{
  get_time_now(*abs_sec, *abs_nsec);
  *abs_nsec += rel_nsec;
  *abs_sec  += rel_sec + *abs_nsec / 1000000000;
  *abs_nsec  = *abs_nsec % 1000000000;
}


int
omni_thread::nt_priority(priority_t pri)
{
  switch (pri) {

  case PRIORITY_LOW:
    return THREAD_PRIORITY_LOWEST;

  case PRIORITY_NORMAL:
    return THREAD_PRIORITY_NORMAL;

  case PRIORITY_HIGH:
    return THREAD_PRIORITY_HIGHEST;
  }

  throw omni_thread_invalid();
  return 0; /* keep msvc++ happy */
}


static void
get_time_now(omni_time_t::secs_t& abs_sec, omni_time_t::ns_t& abs_nsec)
{
  FILETIME ft;
  GetSystemTimePreciseAsFileTime(&ft);

  ULARGE_INTEGER ul;
  ul.u.LowPart  = ft.dwLowDateTime;
  ul.u.HighPart = ft.dwHighDateTime;

  // Value in ul.QuadPart is 100 nanosecond intervals since 1 January 1601.
  _CORBA_ULongLong units_since_epoch = ul.QuadPart - 116444736000000000ULL;

  abs_sec  = units_since_epoch / 10000000;
  abs_nsec = (units_since_epoch % 10000000) * 100;
}

void
omni_thread::stacksize(unsigned long sz)
{
  stack_size = sz;
}

unsigned long
omni_thread::stacksize()
{
  return stack_size;
}

//
// Dummy thread
//

class omni_thread_dummy : public omni_thread {
public:
  inline omni_thread_dummy() : omni_thread()
  {
    _dummy = 1;
    _state = STATE_RUNNING;

    if (!DuplicateHandle(GetCurrentProcess(), GetCurrentThread(),
                         GetCurrentProcess(), &handle,
                         0, FALSE, DUPLICATE_SAME_ACCESS))
      throw omni_thread_fatal(GetLastError());

    nt_id = GetCurrentThreadId();

    if (!TlsSetValue(self_tls_index, (LPVOID)this))
      throw omni_thread_fatal(GetLastError());
  }
  inline ~omni_thread_dummy()
  {
  }
};

omni_thread*
omni_thread::create_dummy()
{
  if (omni_thread::self())
    throw omni_thread_invalid();

  return new omni_thread_dummy;
}

void
omni_thread::release_dummy()
{
  omni_thread* self = omni_thread::self();
  if (!self || !self->_dummy)
    throw omni_thread_invalid();

  TlsSetValue(self_tls_index, (LPVOID)0);

  omni_thread_dummy* dummy = (omni_thread_dummy*)self;
  delete dummy;
}

void
omni_thread::release_finished_dummy(omni_thread* d)
{
  omni_thread_dummy* dummy = (omni_thread_dummy*)d;
  delete dummy;
}


#if defined(__DMC__) && defined(_WINDLL)
BOOL WINAPI DllMain(HINSTANCE hinstDLL, DWORD fdwReason, LPVOID lpvReserved)
{
  return TRUE;
}
#endif
