//				Package : omnithread
// omnithread/posix.h		Created : 7/94 tjr
//
//    Copyright (C) 2002-2025 Apasphere Ltd
//    Copyright (C) 1994-1997 Olivetti & Oracle Research Laboratory
//
//    This file is part of the omnithread library
//
//    The omnithread library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Lesser General Public
//    License as published by the Free Software Foundation; either
//    version 2.1 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public
//    License along with this library. If not, see http://www.gnu.org/licenses/
//
//
// OMNI thread implementation classes for posix threads
//

#ifndef __omnithread_posix_h_
#define __omnithread_posix_h_

#if defined(__alpha__) && defined(__osf1__) || defined(__hpux__)
// stop unnecessary definitions of TRY, etc on OSF
#ifndef EXC_HANDLING
#define EXC_HANDLING
#endif
#endif

#include <pthread.h>

extern "C" void* omni_thread_wrapper(void* ptr);



///////////////////////////////////////////////////////////////////////////
//
// Mutex
//
///////////////////////////////////////////////////////////////////////////

class omni_mutex {
public:
  omni_mutex();
  ~omni_mutex();

  inline void lock()    { pthread_mutex_lock(&posix_mutex);            }
  inline void unlock()  { pthread_mutex_unlock(&posix_mutex);          }

  inline int  trylock() { return !pthread_mutex_trylock(&posix_mutex); }
  // if mutex is unlocked then lock it and return 1 (true).
  // If it is already locked then return 0 (false).

  friend class omni_condition;

private:
  // dummy copy constructor and operator= to prevent copying
  omni_mutex(const omni_mutex&);
  omni_mutex& operator=(const omni_mutex&);

  pthread_mutex_t posix_mutex;
};


///////////////////////////////////////////////////////////////////////////
//
// Condition variable
//
///////////////////////////////////////////////////////////////////////////

class omni_condition {
public:
  omni_condition(omni_mutex* m);
  // constructor must be given a pointer to an existing mutex. The
  // condition variable is then linked to the mutex, so that there is an
  // implicit unlock and lock around wait() and timed_wait().

  ~omni_condition();

  void wait();
  // wait for the condition variable to be signalled.  The mutex is
  // implicitly released before waiting and locked again after waking up.
  // If wait() is called by multiple threads, a signal may wake up more
  // than one thread.  See POSIX threads documentation for details.

  int timedwait(omni_time_t::secs_t secs, omni_time_t::ns_t nanosecs = 0);
  // timedwait() is given an absolute time to wait until.  To wait for a
  // relative time from now, use omni_thread::get_time. See POSIX threads
  // documentation for why absolute times are better than relative.
  // Returns 1 (true) if successfully signalled, 0 (false) if time
  // expired.

  inline int timedwait(const omni_time_t& t)
  {
    return timedwait(t.s, t.ns);
  }

  void signal();
  // if one or more threads have called wait(), signal wakes up at least
  // one of them, possibly more.  See POSIX threads documentation for
  // details.

  void broadcast();
  // broadcast is like signal but wakes all threads which have called
  // wait().

private:
  // dummy copy constructor and operator= to prevent copying
  omni_condition(const omni_condition&);
  omni_condition& operator=(const omni_condition&);

  omni_mutex*    mutex;
  pthread_cond_t posix_cond;
};


#define OMNI_THREAD_IMPLEMENTATION			\
    pthread_t posix_thread;				\
    static int posix_priority(priority_t);		\
    friend void* omni_thread_wrapper(void* ptr);

#endif // __omnithread_posix_h_
