//				Package : omnithread
// omnithread/posix.h		Created : 2023-09-25 dgrisby
//
//    Copyright (C) 2002-2025 Apasphere Ltd
//
//    This file is part of the omnithread library
//
//    The omnithread library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Lesser General Public
//    License as published by the Free Software Foundation; either
//    version 2.1 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public
//    License along with this library. If not, see http://www.gnu.org/licenses/
//

#ifndef __omnithread_base_h_
#define __omnithread_base_h_


///////////////////////////////////////////////////////////////////////////
//
// Exceptions
//
///////////////////////////////////////////////////////////////////////////

//
// This exception is thrown in the event of a fatal error.
//

class _OMNITHREAD_NTDLL_ omni_thread_fatal {
public:
  int error;
  omni_thread_fatal(int e = 0) : error(e) {}
};


//
// This exception is thrown when an operation is invoked with invalid
// arguments.
//

class _OMNITHREAD_NTDLL_ omni_thread_invalid {};


///////////////////////////////////////////////////////////////////////////
//
// Time
//
///////////////////////////////////////////////////////////////////////////

struct omni_time_t {
  typedef _CORBA_ULongLong secs_t; // type for seconds since Unix epoch
  typedef _CORBA_ULong     ns_t;   // type for nanoseconds

  secs_t s;  // seconds
  ns_t   ns; // nanoseconds

  inline omni_time_t(secs_t _s=0, ns_t _ns=0)
    : s(_s), ns(_ns) {}

  inline void assign(secs_t _s=0, ns_t _ns=0)
  {
    s  = _s;
    ns = _ns;
  }

  // void* operator used to test for non-zero time. We cant use bool
  // in case we're using an ancient compiler with no bool.
  inline operator void*() const
  {
    return (void*)(s || ns);
  }

  inline int operator==(const omni_time_t& o) const
  {
    return o.s == s && o.ns == ns;
  }

  inline int operator<(const omni_time_t& o) const
  {
    return s < o.s || (s == o.s && ns < o.ns);
  }

  inline int operator<=(const omni_time_t& o) const
  {
    return s < o.s || (s == o.s && ns <= o.ns);
  }

  inline int operator>(const omni_time_t& o) const
  {
    return s > o.s || (s == o.s && ns > o.ns);
  }

  inline int operator>=(const omni_time_t& o) const
  {
    return s > o.s || (s == o.s && ns >= o.ns);
  }

  inline omni_time_t& operator+=(const omni_time_t& o)
  {
    secs_t os  = o.s + o.ns / 1000000000;
    ns_t   ons = o.ns % 1000000000;
    ns_t   nns = ns + ons;
    s += os + nns / 1000000000;
    ns = nns % 1000000000;
    return *this;
  }

  inline omni_time_t operator+(const omni_time_t& o) const
  {
    omni_time_t n(*this);
    n += o;
    return n;
  }

  inline omni_time_t& operator-=(const omni_time_t& o)
  {
    secs_t os  = o.s + o.ns / 1000000000;
    ns_t   ons = o.ns % 1000000000;
    ns_t   nns = ns + 1000000000 - ons;

    s -= os + (1 - nns / 1000000000);
    ns = nns % 1000000000;
    return *this;
  }

  inline omni_time_t operator-(const omni_time_t& o) const
  {
    omni_time_t n(*this);
    n -= o;
    return n;
  }
};


#endif // __omnithread_base_h_
