#ifndef __VXTHREAD_H__
#define __VXTHREAD_H__
/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Project:     omniORB
%% Filename:    $Filename$
%% Author:      Guillaume/Bill ARRECKX
%%              Copyright Wavetek Wandel & Goltermann, Plymouth.
%% Description: OMNI thread implementation classes for VxWorks threads
%% Notes:
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
*/


///////////////////////////////////////////////////////////////////////////
// Includes
///////////////////////////////////////////////////////////////////////////
#include <vxWorks.h>
#include <semLib.h>
#include <taskLib.h>


///////////////////////////////////////////////////////////////////////////
// Extern prototypes
///////////////////////////////////////////////////////////////////////////

extern "C" void omni_thread_wrapper(void* ptr);



///////////////////////////////////////////////////////////////////////////
//
// Mutex
//
///////////////////////////////////////////////////////////////////////////

class omni_mutex {
public:
  omni_mutex();
  ~omni_mutex();

  inline void lock()
  {
    if (semTake(mutexID, WAIT_FOREVER) != OK){
      throw omni_thread_fatal(errno);
    }
  }

  inline void unlock()
  {
    if (semGive(mutexID) != OK) {
      throw omni_thread_fatal(errno);
    }
  }

  inline int  trylock() {
    // if mutex is unlocked then lock it and return 1 (true).
    // If it is already locked then return 0 (false).

    return semTake(mutexID, NO_WAIT) == OK;
  }

  friend class omni_condition;

private:
  // dummy copy constructor and operator= to prevent copying
  omni_mutex(const omni_mutex&);
  omni_mutex& operator=(const omni_mutex&);

  SEM_ID mutexID;
  bool   m_bConstructed;
};


///////////////////////////////////////////////////////////////////////////
//
// Condition variable
//
///////////////////////////////////////////////////////////////////////////

class omni_condition {
public:
  omni_condition(omni_mutex* m);
  // constructor must be given a pointer to an existing mutex. The
  // condition variable is then linked to the mutex, so that there is an
  // implicit unlock and lock around wait() and timed_wait().

  ~omni_condition();

  void wait();
  // wait for the condition variable to be signalled.  The mutex is
  // implicitly released before waiting and locked again after waking up.
  // If wait() is called by multiple threads, a signal may wake up more
  // than one thread.  See POSIX threads documentation for details.

  int timedwait(omni_time_t::secs_t secs, omni_time_t::ns_t nanosecs = 0);
  // timedwait() is given an absolute time to wait until.  To wait for a
  // relative time from now, use omni_thread::get_time. See POSIX threads
  // documentation for why absolute times are better than relative.
  // Returns 1 (true) if successfully signalled, 0 (false) if time
  // expired.

  inline int timedwait(const omni_time_t& t)
  {
    return timedwait(t.s, t.ns);
  }

  void signal();
  // if one or more threads have called wait(), signal wakes up at least
  // one of them, possibly more.  See POSIX threads documentation for
  // details.

  void broadcast();
  // broadcast is like signal but wakes all threads which have called
  // wait().

private:
  // dummy copy constructor and operator= to prevent copying
  omni_condition(const omni_condition&);
  omni_condition& operator=(const omni_condition&);

  omni_mutex* mutex;
  SEM_ID sema_;
};



///////////////////////////////////////////////////////////////////////////
//
// Counting semaphore
//
///////////////////////////////////////////////////////////////////////////

#define OMNI_HAVE_PLATFORM_SEMAPHORE

class _OMNITHREAD_NTDLL_ omni_semaphore {
public:
  omni_semaphore(unsigned int initial = 1);
  ~omni_semaphore();

  void wait();
  // if semaphore value is > 0 then decrement it and carry on. If it's
  // already 0 then block.

  int trywait();
  // if semaphore value is > 0 then decrement it and return 1 (true).
  // If it's already 0 then return 0 (false).

  void post();
  // if any threads are blocked in wait(), wake one of them up. Otherwise
  // increment the value of the semaphore.

private:
  // dummy copy constructor and operator= to prevent copying
  omni_semaphore(const omni_semaphore&);
  omni_semaphore& operator=(const omni_semaphore&);

  SEM_ID semID;
};



///////////////////////////////////////////////////////////////////////////
// Exported macros
// Note: These are added as private members in each class implementation.
///////////////////////////////////////////////////////////////////////////

#define OMNI_THREAD_IMPLEMENTATION \
  friend void omni_thread_wrapper(void* ptr); \
  static int vxworks_priority(priority_t); \
  omni_condition *running_cond; \
  void* return_val; \
  int tid; \
  public: \
  static void attach(void); \
  static void detach(void); \
  static void show(void);


///////////////////////////////////////////////////////////////////////////
// Porting macros
///////////////////////////////////////////////////////////////////////////
// This is a wrapper function for the 'main' function which does not exists
//  as such in VxWorks. The wrapper creates a launch function instead,
//  which spawns the application wrapped in a omni_thread.
// Argc will always be null.
///////////////////////////////////////////////////////////////////////////
#define main( discarded_argc, discarded_argv ) \
        omni_discard_retval() \
          { \
          throw; \
          } \
        int omni_main( int argc, char **argv ); \
        void launch( ) \
          { \
          omni_thread* th = new omni_thread( (void(*)(void*))omni_main );\
          th->start();\
          }\
        int omni_main( int argc, char **argv )


#endif // ndef __VXTHREAD_H__
